// CPM100U.h : CPM100U DLL header file
//
#ifndef __CPM100_H
#define __CPM100_H

#ifdef __cplusplus
extern "C" {
#endif

#ifdef CPM100_API_EXPORTS
	#define CPM100_API extern "C" __declspec(dllexport)
#else
	#define CPM100_API extern "C" __declspec(dllimport)
#endif

// CANopen Master Error Code Define
#define CPM_NoError         0
#define CPM_DriverError     1

#define CPM_BoardNumberErr  3

#define CPM_ConfigErr       5
#define CPM_MasterInitErr   6
#define CPM_MasterNotInit   7
#define CPM_ListenMode      8
#define CPM_NodeErr         9
#define CPM_NodeExist       10
#define CPM_AddModeErr		  11
#define CPM_TxBusy          12
#define CPM_UnknowCmd       13
#define CPM_CmdReceErr      14
#define CPM_DataEmpty       15
#define CPM_MemAllocErr     16
#define CPM_SendCycMsgErr   17
#define CPM_StatusErr       18

#define CPM_SetGuardErr     20 
#define CPM_SetHbeatErr     21 
#define CPM_SegLenErr       22
#define CPM_SegToggleErr    23
#define CPM_SegWriteErr     24 
#define CPM_Abort           25
#define CPM_PDOLenErr       26 
#define CPM_COBIDErr        27
#define CPM_PDOInstErr      28
#define CPM_PDODynaErr      29
#define CPM_PDONumErr       30
#define CPM_PDOSetErr       31
#define CPM_PDOEntryErr     32
#define CPM_SetCobIdErr     33
#define CPM_CycFullErr      34
#define CPM_Timeout         35
#define CPM_DataLenErr		36
#define CPM_SyncIdDifferent 37

#define CPM_Wait			40
#define CPM_Processing		41

#define CPM_LoadEDSErr		50
#define CPM_EDSFormatErr	51

// CANopen Master NMTErr Event Define
#define CPM_Node_Guarding_Event		1
#define CPM_Heartbeat_Event			2

// SYNC Group Send Stop State
#define CPM_SYNC_GROUP_COMPLETE		1
#define CPM_SYNC_GROUP_FAIL			2

////////////////////////////////////////
//	PISO-CPM100U function library
////////////////////////////////////////

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetVersion
// Description: Return PISO-CPM100(U).dll version
// Parameter: void
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetVersion(void);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_TotalBoard
// Description: Get total PISO-CPM100(U) board number
// Parameter: void
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_TotalBoard(void);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetBoardSwitchNo
// Description: Get PISO-CPM100(U) switch number
// Parameter: bBoardCntNo: PISO-CPM100(U) board queue number
//            *BoardNo: Board switch No of the PISO-CPM100(U) Board
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetBoardSwitchNo(BYTE bBoardCntNo, BYTE *BoardNo);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetBoardInf
// Description: Get PISO-CPM100(U) board information
// Parameter: BoardNo: PISO-CPM100(U) board number.
//            *dwVID: Vendor ID
//			  *dwDID: Device ID
//			  *dwSVID: Sub-vendor ID
//		      *dwSDID: Sub-device ID
//			  *dwSAuxID: Sub-auxiliary ID
//			  *dwIrqNo: IRQ number
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetBoardInf(BYTE BoardNo, DWORD *dwVID,DWORD *dwDID,
						    DWORD *dwSVID,DWORD *dwSDID,DWORD *dwSAuxID,DWORD *dwIrqNo);

////////////////////////////////////////
// Driver Version: 2.01
// Firmware Version: 2.01
// Function: CPM100_SetFunctionTimeout
// Description: Set function timeout
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  FunTimeout: Max timeout ms of per function (default value = 1000 ms).
////////////////////////////////////////
CPM100_API void CALLBACK CPM100_SetFunctionTimeout(BYTE BoardNo, DWORD FunTimeout);

////////////////////////////////////////
// Driver Version: 2.01
// Firmware Version: 2.01
// Function: CPM100_GetCANStatus
// Description: Get CAN error status of the PISO-CPM100(U) board
// Parameter: BoardNo: PISO-CPM100(U) board number.
//            *bStatus: get CAN error status 
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetCANStatus(BYTE BoardNo, BYTE *bStatus);

///////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_InitMaster
// Description: Initial Master, must be call once before all others function
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: 0 is for normal Master, 1 ~ 127 are for Master of slave-mode.
//					The slave-mode Master has node id and can be controled by other CANopen device.
//					We have provided some ISR function for user to do this.
//			  BaudRate: 0 ~ 7 ==> 10 k ~ 1000 k bps
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_InitMaster(BYTE BoardNo, BYTE Node, BYTE BaudRate, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_ShutdownMaster
// Description: Let PISO-CPM100(U) shutdown (reset)
// Parameter: BoardNo: PISO-CPM100(U) board number.
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_ShutdownMaster(BYTE BoardNo);

////////////////////////////////////////
// Driver Version: 2.02
// Firmware Version: 2.02
// Function: CPM100_MasterSendBootupMsg
// Description: Let PISO-CPM100(U) send a boot up message (only for Master Node >= 1)
// Parameter: BoardNo: PISO-CPM100(U) board number.
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_MasterSendBootupMsg(BYTE BoardNo, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_SetMasterMode
// Description: Set Master to normal mode or listen mode
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Mode: 0 is normal mode (default), 1 is listen mode (default)
//					If set the Master to listen mode, the Master will can't send any CANopen message
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SetMasterMode(BYTE BoardNo, BYTE Mode, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetMasterMode
// Description: Get Master mode
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      *Mode: 0 is normal mode (default), 1 is listen mode
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetMasterMode(BYTE BoardNo, BYTE *Mode, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetFirmwareVersion
// Description: Get PISO-CPM100(U) firmware version
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  *Fir_Ver: PISO-CPM100(U) firmware version
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetFirmwareVersion(BYTE BoardNo, WORD *Fir_Ver, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.01
// Firmware Version: 2.01
// Function: CPM100_EDS_Load
// Description: Load EDS File
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node
//			  *FilePath: EDS file path (ex: "\\Temp\\ABC.EDS")
//			  DelayTime: After send a CAN message will delay some time
//			  ResTimeout: Set CAN message timeout
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_EDS_Load(BYTE BoardNo, BYTE Node, char* FilePath, WORD DelayTime, WORD ResTimeout, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00 (AddMode 3 is for 2.01)
// Firmware Version: 2.01
// Function: CPM100_AddNode
// Description: Add Slave Node to Master
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node
//            AddMode: 1:Auto Add Node, 2:Add Manual, 3:Wait to Add
//			  DelayTime: After send a CAN message will delay some time
//			  ResTimeout: Set CAN message timeout
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_AddNode(BYTE BoardNo, BYTE Node, BYTE AddMode, WORD DelayTime, WORD ResTimeout, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_RemoveNode
// Description: Remove Slave Node from Master
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_RemoveNode(BYTE BoardNo, BYTE Node, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.02
// Firmware Version: 2.02
// Function: CPM100_RemoveAndResetNode
// Description: Remove Slave and then Reset Node from Master
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_RemoveAndResetNode(BYTE BoardNo, BYTE Node, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.03
// Firmware Version: 2.02
// Function: CPM100_DelayAndResponseTimeout
// Description: Change delay time and response timeout value of the node
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  DelayTime: After send a CAN message will delay some time
//			  ResTimeout: Set CAN message timeout 
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_DelayAndResponseTimeout(BYTE BoardNo, BYTE Node, WORD DelayTime, WORD ResTimeout, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_ScanNode
// Description: Scan real node id on the CANopen bus
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      S_Node: Start scan node id
//			  E_Node: End scan node id
//					  If S_Node > E_Node or E_Node = 0, will scan all node id (1~127)
//			  *NodeList: 16 BYTE Array. Return Node Id List of the CANopen Bus. Each bit represents a slave node
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_ScanNode(BYTE BoardNo, BYTE S_Node, BYTE E_Node, BYTE *NodeList, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetNodeList
// Description: Get Node Id List of the Master had added
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  *NodeList: 16 BYTE Array. Return Node Id List of the Master had added. Each bit represents a slave node
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetNodeList(BYTE BoardNo, BYTE *NodeList, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_NMTChangeState
// Description: Set Node State
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node Id
//			  State: NMT State 
//			        (1:Operation mode, 2:Stop mode, 128:Pre-operation mode, 129:Reset node, 130:Reset communication)
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_NMTChangeState(BYTE BoardNo, BYTE Node, BYTE State, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_NMTGetState
// Description: Get Node State
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node Id
//			  *State: Return NMT State
//					 (4:Stop mode, 5:Operation mode, 127:Pre-operation mode)
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_NMTGetState(BYTE BoardNo, BYTE Node, BYTE *State, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_NMTGuarding
// Description: Start Guarding
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node Id
//			  GuardTime: set index 0x100C 
//						(Master will send Guarding message per GuardTime ms).
//			  LiftTime: set index 0x100D 
//						(If Master doesn't receive Guarding response over LiftTime times, it will timeout and Node_Guarding_Event occur)
//						(The Guarding timeout = GuardTime * LiftTime )
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
// Note 1: It is not allowed for one slave to use both Guarding protocol and Heartbeat protocol at the same time. 
// Node 2: In listen mode, the function will listen the Guarding message send from other Master to slave on the CANopen bus. 
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_NMTGuarding(BYTE BoardNo, BYTE Node, WORD GuardTime, BYTE LiftTime, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_NMTHeartbeat
// Description: Start Heartbeat
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node Id
//			  ProduceTime: set index 0x1017
//			  ConsumeTime: Timeout of Heartbeat
//						   (ConsumeTime must more then ProduceTime)
//						   (If Master doesn't receive slave Heartbeat over ConsumeTime ms, it will timeout and Heartbeat_Event occur)
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
// Note: It is not allowed for one slave to use both Guarding protocol and Heartbeat protocol at the same time.
// Node 2: In listen mode, the function will listen only and can't start Heartbeat protocol. So Heartbeat protocol must start from other Master. 
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_NMTHeartbeat(BYTE BoardNo, BYTE Node, WORD ProduceTime, WORD ConsumeTime, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_SDOReadData
// Description: SDO Read Data. If the data length is more than 1024 bytes, the Data will return NULL,
//				and user need uses CPM100_SDOReadFile to read the SDO data.
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node Id
//			  Index: Index of Object Dectionary
//			  SubIndex: SubIndex of Object Dectionary
//			  *RDLen: Response Data Length
//			  *RData: Response Data (If RDLen >= 1024, this parameter will return NULL.)
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SDOReadData(BYTE BoardNo, BYTE Node, WORD Index, BYTE SubIndex, DWORD *RDLen, BYTE *RData, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.01
// Firmware Version: 2.01
// Function: CPM100_SDOReadFile
// Description: SDO Read File. After CPM100_SDOReadData, if the data length is more than 1024 bytes,
//				user need uses this function to read the SDO data file.
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node
//			  *Index: Index of Object Dectionary
//			  *SubIndex: SubIndex of Object Dectionary
//			  Start: Start position to read the SDO data file
//			  Len: Max read byte
//			  RDLen: Return read data length
//			  RData: Return data
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SDOReadFile(BYTE BoardNo, BYTE Node, WORD Index, BYTE SubIndex, DWORD Start, DWORD Len, DWORD *RDLen, BYTE *RData);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_SDOWriteData
// Description: SDO Write Data
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node Id
//			  Index: Index of Object Dectionary
//			  SubIndex: SubIndex of Object Dectionary
//			  TDLen: Write Data Length
//			  *TData: Write Data
//			  *RDLen: Response Data Length
//			  *RData: Response Data
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SDOWriteData(BYTE BoardNo, BYTE Node, WORD Index, BYTE SubIndex, DWORD TDLen, BYTE *TData, WORD *RDLen, BYTE *RData, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_SDOAbortTransmit
// Description: Send SDO Abort Message
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  Index: Index of Object Dectionary
//			  SubIndex: SubIndex of Object Dectionary
//			  *TData:  Abort Data
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SDOAbortTransmit(BYTE BoardNo, BYTE Node, WORD Index, BYTE SubIndex, DWORD TData, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_PDOWrite
// Description: Output PDO data
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Cobid: PDO Cob Id
//			  DLen: Data Length
//			  *Data: PDO Data
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_PDOWrite(BYTE BoardNo, WORD Cobid, BYTE Offset, BYTE DLen, BYTE *Data, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.02
// Firmware Version: 2.02
// Function: CPM100_PDOWrite_Fast
// Description: Output PDO data directly
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Cobid: PDO Cob Id
//			  DLen: Data Length
//			  Data: PDO Data
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_PDOWrite_Fast(BYTE BoardNo, WORD Cobid, BYTE Offset, BYTE DLen, BYTE *Data);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_PDORemote
// Description: Remote PDO data
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Cobid: PDO Cob Id
//			  *DLen: Response Data Length
//			  *Data: Response PDO Data
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_PDORemote(BYTE BoardNo, WORD Cobid, BYTE *DLen, BYTE *Data, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.02
// Firmware Version: 2.02
// Function: CPM100_PDORemote_Fast
// Description: Send requst to remote PDO data, but do not check the response
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Cobid: PDO Cob Id
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_PDORemote_Fast(BYTE BoardNo, WORD Cobid);

////////////////////////////////////////
// Driver Version: 2.01
// Firmware Version: 2.01
// Function: CPM100_SetPDORemotePolling
// Description: Set PDO polling list for remote PDO (max 125 PDO one time)
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      PDOCnt: Total PDO number that want to be polling 
//			  Cobid: PDO list that want to be polling
//			  PollingTime: Min. polling Time
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SetPDORemotePolling(BYTE BoardNo, BYTE PDOCnt, WORD *Cobid, WORD PollingTime, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetPDOLastData
// Description: Get PDO last data
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Cobid: PDO Cob Id
//			  *IsNew: Is new data, 0 is old data, 1 is new data
//			  *DLen: Get PDO Data Length
//			  *Data: Get PDO Data
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetPDOLastData(BYTE BoardNo, WORD Cobid, BYTE *IsNew, BYTE *DLen, BYTE *Data, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.02
// Firmware Version: 2.02
// Function: CPM100_GetPDOLastData_Fast
// Description: Get PDO data from DPRAM directly
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Cobid: PDO Cob Id
//			  *DLen: PDO Data Length
//			  *Data: PDO Data
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetPDOLastData_Fast(BYTE BoardNo, WORD Cobid, BYTE *DLen, BYTE *Data);

////////////////////////////////////////
// Driver Version: 2.01
// Firmware Version: 2.01
// Function: CPM100_GetMultiPDOData
// Description: Get multi PDO data (this function can get max 50 PDO data every times)
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      PDOCnt: Total PDO number that want to get 
//		      Cobid: All PDO Cob Id that want to get
//			  IsNew: Is new data, 0 is old data, 1 is new data
//			  DLen: PDO Data Length
//			  Data: PDO Data
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetMultiPDOData(BYTE BoardNo, BYTE PDOCnt, WORD *Cobid, BYTE *IsNew, BYTE *DLen, BYTE *Data, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.02
// Firmware Version: 2.02
// Function: CPM100_GetMultiPDOData_Fast
// Description: Get multi PDO data from DPRAM directly
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      PDOCnt: Total PDO number that want to get 
//		      Cobid: All PDO Cob Id that want to get
//			  DLen: PDO Data Length
//			  Data: PDO Data
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetMultiPDOData_Fast(BYTE BoardNo, BYTE PDOCnt, WORD *Cobid, BYTE *DLen, BYTE *Data);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetRxPDOID
// Description: Get Rx PDO Cob ID List of slave
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  *PDO_Cnt: Response RxPDO ID total number
//			  *Id_List: Response *PDO_Cnt number RxPDO ID List
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetRxPDOID(BYTE BoardNo, BYTE Node, BYTE *PDO_Cnt, WORD *Id_List, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetTxPDOID
// Description: Get Tx PDO Cob ID List of slave
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  *PDO_Cnt: Response TxPDO ID total number
//			  *Id_List: Response *PDO_Cnt number TxPDO ID List
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetTxPDOID(BYTE BoardNo, BYTE Node, BYTE *PDO_Cnt, WORD *Id_List, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_InstallPDO
// Description: Install PDO ID to Master
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  Cobid: PDO Cob ID
//			  RxTx: 0 is RxPDO, 1 is TxPDO 
//			  PDO_No: PDO No. 0 ~ 511
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_InstallPDO(BYTE BoardNo, BYTE Node, WORD Cobid, BYTE RxTx, WORD PDO_No, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_DynamicPDO
// Description: Dynamic PDO Mapping to Master
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  Cobid: PDO Cob ID
//			  RxTx: 0 is RxPDO, 1 is TxPDO 
//			  Entry: Mapping Entry
//			  EntryData: Mapping Data
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_DynamicPDO(BYTE BoardNo, BYTE Node, WORD Cobid, BYTE RxTx, BYTE Entry, DWORD EntryData, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_RemovePDO
// Description: Remove POD Entry
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  Cobid: PDO Cob ID
//			  Entry: Remove Entry, 0 is for whole PDO
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_RemovePDO(BYTE BoardNo, BYTE Node, WORD Cobid, BYTE Entry, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_ChangePDOID
// Description: Change POD ID
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Old_Cobid: Old Cobid
//			  New_Cobid: Want to change to New Cobid
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_ChangePDOID(BYTE BoardNo, WORD Old_Cobid, WORD New_Cobid, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetPDOMapInfo
// Description: Get PDO mapping information
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Cobid: PDO Cobid
//			  *PDONo: Get PDO No.
//			  *RxTx: 0 is RxPDO, 1 is TxPDO
//			  *Tx_Type: Transmission Type
//			  *Event_Timer: Event Timer
//			  *Entry_Cnt: Total entry number
//			  *Map_Data: Mapping Data
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetPDOMapInfo(BYTE BoardNo, WORD Cobid, WORD *PDONo, BYTE *RxTx, BYTE *Tx_Type, WORD *Event_Timer, BYTE *Entry_Cnt, DWORD *Map_Data, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_InstallPDO_List (Virtual List)
// Description: Install PDO and Mapping PDO Data to Master List.
//				This function like CPM100_InstallPDO_List, but it only build a virtual PDO let Master
//				can control the PDO. The virtual PDO is not the real node slave necessarily.
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  Cobid: PDO Cob ID
//			  RxTx: 0 is RxPDO, 1 is TxPDO
//			  PDO_No: PDO No. 
//			  TransmitType: Set transmission type
//			  EventTimer: Set event timer
//			  EntryUse: Total Mapping Entry
//			  *EntryData: DWORD array for Total Mapping Data
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_InstallPDO_List(BYTE BoardNo, BYTE Node, WORD Cobid, BYTE RxTx, WORD PDO_No, BYTE TransmitType, WORD EventTimer, BYTE EntryUse, DWORD *EntryData, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_RemovePDO_List (Virtual List)
// Description: Remove PDO build from CPM100_InstallPDO_List function
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  Cobid: PDO Cob ID
//			  Entry: Remove Entry, 0 is for whole PDO
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_RemovePDO_List(BYTE BoardNo, BYTE Node, WORD Cobid, BYTE Entry, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_PDOUseEntry
// Description: Change useful entry
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Cobid: PDO Cob ID
//			  Entry: useful entry
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_PDOUseEntry(BYTE BoardNo, WORD Cobid, BYTE Entry, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_PDOTxType
// Description: Set PDO Transmission Type
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Cobid: PDO Cob ID
//			  Tx_Type: Transmission Type
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_PDOTxType(BYTE BoardNo, WORD Cobid, BYTE Tx_Type, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_PDOEventTimer
// Description: Set PDO Event Timer
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Cobid: PDO Cob ID
//			  Timer: Event Timer
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_PDOEventTimer(BYTE BoardNo, WORD Cobid, WORD Timer, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.03
// Firmware Version: 2.02
// Function: CPM100_PDOInhibitTime
// Description: Set PDO Inhibit Time
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Cobid: PDO Cob ID
//			  Timer: Inhibit Time
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_PDOInhibitTime(BYTE BoardNo, WORD Cobid, WORD Time, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_ChangeSYNCID
// Description: Change SYNC ID
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node Id
//			  Cobid: SYNC ID
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_ChangeSYNCID(BYTE BoardNo, BYTE Node, WORD Cobid, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_SetSYNC_List (Virtual List)
// Description: Set or Change SYNC ID from Master
//				The function only build a virtual SYNC ID let Master can control it. 
//				The virtual SYNC ID is not the real one necessarily.
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  Cobid: PDO Cob ID
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SetSYNC_List(BYTE BoardNo, BYTE Node, WORD Cobid, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetSYNCID
// Description: Get SYNC ID
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node Id
//			  *Cobid: SYNC ID
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetSYNCID(BYTE BoardNo, BYTE Node, WORD *Cobid, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_SendSYNCMsg
// Description: Send SYNC message, most send 5 different SYNC at the same time
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Cobid: SYNC ID
//			  Timer: Cyclic timer, 0 is stop or send 1 time SYNC message
//			  Times: Cyclic times, 0 is always contiune
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SendSYNCMsg(BYTE BoardNo, WORD Cobid, WORD Timer, DWORD Times, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.05
// Firmware Version: 2.03
// Function: CPM_SyncGroupCycUpdata
// Description: Before use CPM100_SyncGroupCycSend to send SYNC and PDO group data, users should  
//				use this function to fill the Max 15 cycle buffer. If uses want their devices run
//				more group cycles, users can still use this function to contiunously fill the cycle
//				buffer after execute CPM100_SyncGroupCycSend.
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  StepNum: Cycle number of this update mission. This parameter is < 15 ( StepNum * RecordNum < 60 ).
//			  RecordNum: PDO number of this SYNC group. This parameter is < 32. (Every update, the PDO number of the group data must be the same.)
//			  *PDOId: PDO ID Array of these group data (Array type: WORD [RecordNum])
//			  *GroupData: Initial Data of output every PDO (Array type: BYTE [StepNum][RecordNum][8])
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SyncGroupCycUpdata(BYTE BoardNo, BYTE StepNum, BYTE RecordNum, WORD *PDOId, BYTE *GroupData, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.05
// Firmware Version: 2.03
// Function: CPM_SyncGroupCycSend
// Description: Use this function start to Send SYNC and PDO group data from group cycle buffer
//              (Update by CPM_SyncGroupCycUpdata).
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  TotalCycle: Total Cycle for this Sync Group. TotalCycle = 0 means unlimited.
//			  CycInterval: Cycle timer, unit is 0.1 ms and minimum value is 5 (0.5 ms).
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SyncGroupCycSend(BYTE BoardNo, DWORD TotalCycle, DWORD CycInterval, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.05
// Firmware Version: 2.03
// Function: CPM_SyncGroupClear
// Description: Clear All Step of the SYNC Group Send
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
// Return: CPM_NoError, CPM_Timeout
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SyncGroupClear(BYTE BoardNo, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM_GetCyclicSYNCInfo
// Description: Get Cyclic SYNC Info
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  *Cobid: 5 WORD Array, SYNC ID
//			  *Timer: 5 WORD Array, Cyclic timer, 0 is stop or 1 time and
//			  *Times: 5 DWORD Array, Cyclic times, 0 is always contiune
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetCyclicSYNCInfo(BYTE BoardNo, WORD *Cobid, WORD *Timer, DWORD *Times, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_ChangeEMCYID
// Description: Change EMCY ID
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node Id
//			  Cobid: EMCY ID
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_ChangeEMCYID(BYTE BoardNo, BYTE Node, WORD Cobid, BYTE BlockMode);

//////////////////////////////////////// 
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_SetEMCY_List (Virtual List)
// Description: Set or Change EMCY ID from Master
//				The function only build a virtual EMCY ID let Master can control it. 
//				The virtual EMCY ID is not the real one necessarily.
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  Cobid: PDO Cob ID
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_SetEMCY_List(BYTE BoardNo, BYTE Node, WORD Cobid, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetEMCYID
// Description: Get EMCY ID
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  Node: Slave Node
//			  *Cobid: EMCY ID
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetEMCYID(BYTE BoardNo, BYTE Node, WORD *Cobid, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.01
// Firmware Version: 2.01
// Function: CPM100_ReadLastEMCY
// Description: Get last message of the slave node
// Parameter: BoardNo: PISO-CPM100(U) board number.
//		      Node: Slave Node Id
//			  *IsNew: Is new data, 0 is old data, 1 is new data
//			  *Data: EMCY Data
//			  BlockMode: Function Mode: Non-block function mode = 0, Block function mode = 1
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_ReadLastEMCY(BYTE BoardNo, BYTE Node, BYTE *IsNew, BYTE *Data, BYTE BlockMode);

////////////////////////////////////////
// Driver Version: 2.05
// Firmware Version: 2.03
// Function: CPM100_GetSYNCGroupStopState
// Description: Get SYNC Group Send Stop State after SYNC Group Send Stop
// Parameter: *BoardNo: PISO-CPM100(U) board number.
//		      *State: 1: SYNC Group Send Complete, 2: SYNC Group Send Fail
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetSYNCGroupStopState(BYTE *BoardNo, BYTE *State);

////////////////////////////////////////
// Driver Version: 2.02
// Firmware Version: 2.02
// Function: CPM100_GetBootUpNodeAfterAdd
// Description: Get Boot up message after add node
// Parameter: *BoardNo: PISO-CPM100(U) board number.
//		      *Node: Slave Node Id
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetBootUpNodeAfterAdd(BYTE *BoardNo, BYTE *Node);

////////////////////////////////////////
// Driver Version: 2.01
// Firmware Version: 2.01
// Function: CPM100_GetEMCYData
// Description: Get EMCY data received from the PISO-CPM100(U)
// Parameter: *BoardNo: PISO-CPM100(U) board number.
//		      *Node: Slave Node Id
//			  *Data: EMCY Data
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetEMCYData(BYTE *BoardNo, BYTE *Node, BYTE *Data);

////////////////////////////////////////
// Driver Version: 2.01
// Firmware Version: 2.01
// Function: CPM100_GetNMTError
// Description: Get NMT Error Control message from CANopen slave. 
// Parameter: *BoardNo: PISO-CPM100(U) board number.
//		      *Node: Slave Node Id
//			  *NMTErrMode: Get NMT Error mode.
//                         If Node_Guarding_Event, Mode = CPM_Node_Guarding_Event.
//                         And if Heartbeat_Event, Mode = CPM_Heartbeat_Event.
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetNMTError(BYTE *BoardNo, BYTE *Node, BYTE *NMTErrMode);

// CPM100_InstallSYNCGroupISR:
// Driver Version: 2.05
// Firmware Version: 2.03
// After called CPM100_InstallSYNCGroupISR to install SYNC Group ISR function.
// When the SYNC Group data is stopped, a stop state event will produce. And than
// the SYNC Group ISR will be executed.
CPM100_API WORD CALLBACK CPM100_InstallSYNCGroupISR(BYTE BoardNo, void (*SYNCISR)());
CPM100_API WORD CALLBACK CPM100_RemoveSYNCGroupISR(BYTE BoardNo);

// CPM100_InstallBootUpISR:
// Driver Version: 2.02
// Firmware Version: 2.02
// After called CPM100_InstallBootUpISR to install Boot ISR function.
// When the Master received a Boot up message send from other slave device witch
// had been added to the Master by calling CPM100_AddNode, the Boot up ISR will be executed.
CPM100_API WORD CALLBACK CPM100_InstallBootUpISR(BYTE BoardNo, void (*BOOTISR)());
CPM100_API WORD CALLBACK CPM100_RemoveBootUpISR(BYTE BoardNo);

// CPM100_InstallRxSDOISR:
// Driver Version: 2.00
// Firmware Version: 2.01
// After called CPM100_InstallEMCYISR to install EMCY ISR function.
// When the Master received a EMCY message send from other slave device witch
// had been added to the Master by calling CPM100_AddNode, the EMCY ISR will be executed.
CPM100_API WORD CALLBACK CPM100_InstallEMCYISR(BYTE BoardNo, void (*EMCYISR)());
CPM100_API WORD CALLBACK CPM100_RemoveEMCYISR(BYTE BoardNo);

// CPM100_InstallNMTErrISR:
// Driver Version: 2.00
// Firmware Version: 2.01
// After called CPM100_InstallNMTErrISR to install NMTErr ISR function.
// When the Master occured a Node_Guarding_Event or Heartbeat_Event, the NMTErr ISR will be executed.
CPM100_API WORD CALLBACK CPM100_InstallNMTErrISR(BYTE BoardNo, void (*NMTERRISR)());
CPM100_API WORD CALLBACK CPM100_RemoveNMTErrISR(BYTE BoardNo);


///////////////////////////////////////////////////////////////////////////////////////////////////
// Following ISR Funtions are only for Master with node id > 0.
// It means if the "Node" parameter of CPM100_InitMaster is 0, the following functions will useless.
///////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetMasterReadSDOEvent
// Description: Get the Read SDO message for Master node id. 
// Parameter: *BoardNo: PISO-CPM100(U) board number.
//		      *Index: Index of Object Dectionary.
//			  *SubIndex: SubIndex of Object Dectionary.
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetMasterReadSDOEvent(BYTE *BoardNo, WORD *Index, BYTE *SubIndex);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetMasterWriteSDOEvent
// Description: Get the Write SDO message for Master node id. 
// Parameter: *BoardNo: PISO-CPM100(U) board number.
//		      *Index: Index of Object Dectionary.
//			  *SubIndex: SubIndex of Object Dectionary.
//			  *WLen: Write Data Length.
//			  *WData: Write Data.
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetMasterWriteSDOEvent(BYTE *BoardNo, WORD *Index, BYTE *SubIndex, BYTE *WLen, BYTE *WData);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_ResponseMasterSDO
// Description: Response SDO message from CPM100U
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  ResType: Response read/write to SDO message (0 is read response, 1 is write response)
//			  Index: Response Index of Object Dectionary
//			  SubIndex: Response SubIndex of Object Dectionary
//			  Len: Response data length (0 ~ 4) (Note: Write response must 0-byte)
//			  *Data: Response data array (max. 4-bytes)
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_ResponseMasterSDO(BYTE BoardNo, BYTE ResType, WORD Index, BYTE SubIndex, BYTE Len, BYTE *Data);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetMasterRemotePDOEvent
// Description: Get the Remote PDO message for Master's PDO COB-ID. 
// Parameter: *BoardNo: PISO-CPM100(U) board number.
//		      *CobId: PDO Cob Id.
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetMasterRemotePDOEvent(BYTE *BoardNo, WORD *CobId);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_GetMasterRxPDOEvent
// Description: Get the Write PDO message for Master's PDO COB-ID. 
// Parameter: *BoardNo: PISO-CPM100(U) board number.
//		      *CobId: PDO Cob Id.
//			  *WLen: Write Data Length.
//			  *WData: Write Data.
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_GetMasterRxPDOEvent(BYTE *BoardNo, WORD *CobId, BYTE *WLen, BYTE *WData);

////////////////////////////////////////
// Driver Version: 2.00
// Firmware Version: 2.01
// Function: CPM100_ResponseMasterPDO
// Description: Response remote PDO message from CPM100U
// Parameter: BoardNo: PISO-CPM100(U) board number.
//			  CobId: Response remote PDO Cob Id.
//			  Len: Response data length (1 ~ 8).
//			  *Data: Response data array (max. 8-bytes).
////////////////////////////////////////
CPM100_API WORD CALLBACK CPM100_ResponseMasterPDO(BYTE BoardNo, WORD CobId, BYTE Len, BYTE *Data);

// CPM100_InstallReadSDOISR:
// Driver Version: 2.00
// Firmware Version: 2.01
// If the Master had called CPM100_InitMaster to initial Master with node n (128 > n > 0).
// After installed CPM100_InstallReadSDOISR, when the Master received a Read RxSDO message (message Cob ID is 0x600+n)
// and the Master will produce an interrupt to execute the RSDOISR ISR function.
CPM100_API WORD CALLBACK CPM100_InstallReadSDOISR(BYTE BoardNo, void (*RSDOISR)());
CPM100_API WORD CALLBACK CPM100_RemoveReadSDOISR(BYTE BoardNo);

// CPM100_InstallWriteSDOISR:
// Driver Version: 2.00
// Firmware Version: 2.01
// If the Master had called CPM100_InitMaster to initial Master with node n (128 > n > 0).
// After installed CPM100_InstallWriteSDOISR, when the Master received a Write RxSDO message (message Cob ID is 0x600+n)
// and the Master will produce an interrupt to execute the WSDOISR ISR function.
CPM100_API WORD CALLBACK CPM100_InstallWriteSDOISR(BYTE BoardNo, void (*WSDOISR)());
CPM100_API WORD CALLBACK CPM100_RemoveWriteSDOISR(BYTE BoardNo);

// CPM100_InstallRxPDOISR:
// Driver Version: 2.00
// Firmware Version: 2.01
// If the Master had called CPM100_InitMaster to initial Master with node n (128 > n > 0).
// And if the Master also had called CPM100_InstallPDO_List to install a RxPDO to itself (ex: PDO Cob ID is 0x200+n).
// After installed CPM100_InstallRxPDOISR, when the Master received the RxPDO message
// and the Master will produce an interrupt to execute the RXPDOISR ISR function.
CPM100_API WORD CALLBACK CPM100_InstallRxPDOISR(BYTE BoardNo, void (*RXPDOISR)());
CPM100_API WORD CALLBACK CPM100_RemoveRxPDOISR(BYTE BoardNo);

// CPM100_InstallRemotePDOISR:
// Driver Version: 2.00
// Firmware Version: 2.01
// If the Master had called CPM100_InitMaster to initial Master with node n (128 > n > 0).
// And if the Master also had called CPM100_InstallPDO_List to install a TxPDO to itself (ex: PDO Cob ID is 0x180+n).
// After installed CPM100_InstallRemotePDOISR, when the Master received the TxPDO message
// and the Master will produce an interrupt to execute the REMOTEPDOISR ISR function.
CPM100_API WORD CALLBACK CPM100_InstallRemotePDOISR(BYTE BoardNo, void (*REMOTEPDOISR)());
CPM100_API WORD CALLBACK CPM100_RemoveRemotePDOISR(BYTE BoardNo);

#ifdef __cplusplus
}
#endif

#endif	// __CPM100_H
